<?php
/**
 * Plugin Name: TheSkinnyAI Widget
 * Plugin URI: https://theskinnyai.com
 * Description: Adds TheSkinnyAI conversational AI assistant to your WordPress site for 24/7 lead qualification and visitor engagement.
 * Version: 1.1
 * Author: TheSkinnyAI
 * Author URI: https://theskinnyai.com
 * License: GPL2
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: theskinnyai
 */

if (!defined('ABSPATH')) exit;

// Add settings page
add_action('admin_menu', 'theskinnyai_add_settings_page');
add_action('admin_init', 'theskinnyai_register_settings');

function theskinnyai_add_settings_page() {
    add_options_page(
        __('TheSkinnyAI Settings', 'theskinnyai'),
        __('TheSkinnyAI', 'theskinnyai'),
        'manage_options',
        'theskinnyai-settings',
        'theskinnyai_options_page'
    );
}

function theskinnyai_register_settings() {
    register_setting('theskinnyai_settings', 'theskinnyai_tenant_id', array(
        'type' => 'string',
        'sanitize_callback' => 'sanitize_text_field',
        'default' => ''
    ));
    register_setting('theskinnyai_settings', 'theskinnyai_site_id', array(
        'type' => 'string',
        'sanitize_callback' => 'sanitize_text_field',
        'default' => ''
    ));
    register_setting('theskinnyai_settings', 'theskinnyai_anon_key', array(
        'type' => 'string',
        'sanitize_callback' => 'sanitize_text_field',
        'default' => ''
    ));
    register_setting('theskinnyai_settings', 'theskinnyai_position', array(
        'type' => 'string',
        'sanitize_callback' => 'sanitize_text_field',
        'default' => 'bottom-right'
    ));

    add_settings_section(
        'theskinnyai_main_section',
        __('Widget Configuration', 'theskinnyai'),
        'theskinnyai_section_callback',
        'theskinnyai-settings'
    );

    add_settings_field(
        'theskinnyai_tenant_id',
        __('Tenant ID', 'theskinnyai') . ' <span style="color:red;">*</span>',
        'theskinnyai_tenant_id_render',
        'theskinnyai-settings',
        'theskinnyai_main_section'
    );

    add_settings_field(
        'theskinnyai_site_id',
        __('Site ID', 'theskinnyai') . ' <span style="color:red;">*</span>',
        'theskinnyai_site_id_render',
        'theskinnyai-settings',
        'theskinnyai_main_section'
    );

    add_settings_field(
        'theskinnyai_anon_key',
        __('Anon Key', 'theskinnyai') . ' <span style="color:red;">*</span>',
        'theskinnyai_anon_key_render',
        'theskinnyai-settings',
        'theskinnyai_main_section'
    );

    add_settings_field(
        'theskinnyai_position',
        __('Widget Position', 'theskinnyai'),
        'theskinnyai_position_render',
        'theskinnyai-settings',
        'theskinnyai_main_section'
    );
}

function theskinnyai_section_callback() {
    echo '<p>' . __('Enter your TheSkinnyAI credentials to enable the AI assistant on your site. You can find these in your <a href="https://theskinnyai.com/dashboard" target="_blank">TheSkinnyAI Dashboard</a> under Site Settings → Widget Code.', 'theskinnyai') . '</p>';
}

function theskinnyai_tenant_id_render() {
    $tenant_id = get_option('theskinnyai_tenant_id', '');
    ?>
    <input type="text" name="theskinnyai_tenant_id" value="<?php echo esc_attr($tenant_id); ?>" placeholder="your-tenant-id" style="width: 400px;">
    <p class="description"><?php _e('Your unique Tenant ID (required)', 'theskinnyai'); ?></p>
    <?php
}

function theskinnyai_site_id_render() {
    $site_id = get_option('theskinnyai_site_id', '');
    ?>
    <input type="text" name="theskinnyai_site_id" value="<?php echo esc_attr($site_id); ?>" placeholder="your-site-id" style="width: 400px;">
    <p class="description"><?php _e('Your Site ID (required)', 'theskinnyai'); ?></p>
    <?php
}

function theskinnyai_anon_key_render() {
    $anon_key = get_option('theskinnyai_anon_key', '');
    ?>
    <input type="text" name="theskinnyai_anon_key" value="<?php echo esc_attr($anon_key); ?>" placeholder="eyJhbGciOiJ..." style="width: 400px;">
    <p class="description"><?php _e('Your Anon Key for API authentication (required)', 'theskinnyai'); ?></p>
    <?php
}

function theskinnyai_position_render() {
    $position = get_option('theskinnyai_position', 'bottom-right');
    ?>
    <select name="theskinnyai_position" style="width: 200px;">
        <option value="bottom-right" <?php selected($position, 'bottom-right'); ?>>Bottom Right</option>
        <option value="bottom-left" <?php selected($position, 'bottom-left'); ?>>Bottom Left</option>
        <option value="top-right" <?php selected($position, 'top-right'); ?>>Top Right</option>
        <option value="top-left" <?php selected($position, 'top-left'); ?>>Top Left</option>
    </select>
    <p class="description"><?php _e('Where the chat widget appears on your site (default: Bottom Right)', 'theskinnyai'); ?></p>
    <?php
}

function theskinnyai_options_page() {
    ?>
    <div class="wrap">
        <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
        
        <div style="background: #fff; padding: 20px; border: 1px solid #ccd0d4; margin: 20px 0;">
            <h2 style="margin-top: 0;">🤖 TheSkinnyAI - AI Sales Assistant</h2>
            <p>TheSkinnyAI adds an intelligent AI assistant to your website that:</p>
            <ul style="list-style: disc; margin-left: 20px;">
                <li>Answers visitor questions 24/7 using your website content</li>
                <li>Qualifies leads and captures contact information</li>
                <li>Reduces bounce rate and increases engagement</li>
            </ul>
            <p><a href="https://theskinnyai.com/signup" target="_blank" class="button button-primary">Get Started Free →</a></p>
        </div>
        
        <form action="options.php" method="post">
            <?php
            settings_fields('theskinnyai_settings');
            do_settings_sections('theskinnyai-settings');
            submit_button(__('Save Settings', 'theskinnyai'));
            ?>
        </form>
        
        <div style="background: #f0f6fc; padding: 15px; border: 1px solid #c3d3e0; margin-top: 20px;">
            <h3 style="margin-top: 0;">📋 Where to Find Your Credentials</h3>
            <ol>
                <li>Log in to your <a href="https://theskinnyai.com/dashboard" target="_blank">TheSkinnyAI Dashboard</a></li>
                <li>Select your site from the list</li>
                <li>Go to <strong>Site Settings → Widget Code</strong></li>
                <li>Copy the values for Tenant ID, Site ID, and Anon Key</li>
            </ol>
        </div>

        <?php
        $tenant_id = get_option('theskinnyai_tenant_id', '');
        $site_id = get_option('theskinnyai_site_id', '');
        $anon_key = get_option('theskinnyai_anon_key', '');
        
        if ($tenant_id && $site_id && $anon_key) {
            echo '<div style="background: #d4edda; padding: 15px; border: 1px solid #c3e6cb; margin-top: 20px;">';
            echo '<strong>✅ Widget Active!</strong> TheSkinnyAI is now running on your site.';
            echo '</div>';
        } elseif ($tenant_id || $site_id || $anon_key) {
            echo '<div style="background: #fff3cd; padding: 15px; border: 1px solid #ffc107; margin-top: 20px;">';
            echo '<strong>⚠️ Missing Required Fields:</strong> Please fill in all three fields (Tenant ID, Site ID, and Anon Key) for the widget to work.';
            echo '</div>';
        }
        ?>
    </div>
    <?php
}

// Inject the widget script in the footer
add_action('wp_footer', 'theskinnyai_inject_widget');

function theskinnyai_inject_widget() {
    $tenant_id = get_option('theskinnyai_tenant_id', '');
    $site_id = get_option('theskinnyai_site_id', '');
    $anon_key = get_option('theskinnyai_anon_key', '');
    $position = get_option('theskinnyai_position', 'bottom-right');
    
    // Only inject if all required fields are present
    if (empty($tenant_id) || empty($site_id) || empty($anon_key)) {
        return;
    }
    ?>
    <script>
    (function(){
        try {
            var loader = document.createElement('script');
            loader.src = 'https://theskinnyai.com/skinny-widget-loader.js';
            loader.dataset.tenantId = <?php echo wp_json_encode($tenant_id); ?>;
            loader.dataset.siteId = <?php echo wp_json_encode($site_id); ?>;
            loader.dataset.anonKey = <?php echo wp_json_encode($anon_key); ?>;
            loader.dataset.widgetHost = 'https://theskinnyai.com';
            loader.dataset.position = <?php echo wp_json_encode($position); ?>;
            document.body.appendChild(loader);
        } catch (e) { /* ignore */ }
    })();
    </script>
    <?php
}

// Add settings link on plugins page
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'theskinnyai_settings_link');

function theskinnyai_settings_link($links) {
    $settings_link = '<a href="options-general.php?page=theskinnyai-settings">' . __('Settings', 'theskinnyai') . '</a>';
    array_unshift($links, $settings_link);
    return $links;
}
